module sortmod

  ! The sortmod Module
  ! ==================
  !
  ! This module uses quicksort to sort arrays (in place) or to return
  ! an index vector that sorts the array. It can sort vectors or 2-D arrays. 
  ! In the latter case, an element is a column of the array, and elements
  ! are compared in the following way, using the `cmp_le` function::
  !
  !    cmp_le( a(:,i), a(:,j) ) =  ( a(k,i) < a(k,j) )
  !
  ! where ``k`` is the first index for which ``a(k,i) /= a(k,j)``. 
  !
  ! Exported Routines
  ! -----------------
  !
  ! call sort( n, a )
  !    sorts a vector `a` of size `n` in place. The vector can be of type
  !    INTEGER, REAL or REAL(8).
  !
  ! call argsort( n, a, idx )
  !    returns the vector `idx` of size `n` that describes the position of
  !    each element of vector `a` (of size `n`) when the array `a` is sorted,
  !    i.e. ``a(idx(1))`` is the smallest element in `a`, ``a(idx(2))`` is the second
  !    smallest element, etc.; the vector `a` can be of type INTEGER, REAL or
  !    REAL(8). The vector `a` is not modified, and `idx` will be initialized
  !    by argsort.
  ! 
  ! To sort 2D arrays, the same routines are available, with one new parameter,
  ! the size of an element (or the number of rows of the array), `m`.
  !
  ! call sort( n, m, a )
  ! call argsort( n, m, a, idx )
  !    to sort/argsort an array `a` of size ``(m,n)``
  ! 
  !
  ! The signature of the comparison function is
  !
  ! function cmp_le( v1, v2, m )
  !    where v1 and v2 are vectors of size `m`.
  !
  ! Two values separated by less than `atol` are considered equal.
  !
  !
  ! Notes
  ! -----
  !
  ! The base sorting routines are adapted from Rosetta Code. The pivot shall be
  ! the middle element of the (sub)vector being currently sorted. The sort is not
  ! stable. No global storage is used. Short lists are sorted by insertion sort.
  ! Performance seems decent.

  implicit none

  private

  ! arrays (or section of arrays) shorter than SST elements will be
  ! sorted by insertion sort
  integer, parameter :: SST = 16


  interface argsort
     module procedure argsort1di
     module procedure argsort1dr
     module procedure argsort1dd
     module procedure argsortmdi
     module procedure argsortmdr
     module procedure argsortmdd
  end interface

  interface sort
     module procedure sort1di_new
     module procedure sort1dr_new
     module procedure sort1dd_new
     module procedure sortmdi_new
     module procedure sortmdr_
     module procedure sortmdd_
  end interface

  interface cmp_le
     module procedure lei
     module procedure ler
     module procedure led
  end interface

  public :: sort, argsort, cmp_le

contains


!================================================================================
subroutine sort1di_new( n, a )
!================================================================================
  integer, intent(in) :: n
  integer, dimension(n), intent(inout) :: a
  call sort1di( n, a, 1, n )
end subroutine sort1di_new

!================================================================================
subroutine sort1dr_new( n, a )
!================================================================================
  integer, intent(in) :: n
  real, dimension(n), intent(inout) :: a
  call sort1dr( n, a, 1, n )
end subroutine sort1dr_new

!================================================================================
subroutine sort1dd_new( n, a )
!================================================================================
  integer, intent(in) :: n
  real(8), dimension(n), intent(inout) :: a
  call sort1dd( n, a, 1, n )
end subroutine sort1dd_new

!================================================================================
subroutine sortmdi_new( n, m, a )
!================================================================================
  integer, intent(in) :: n, m
  integer, dimension(m,n), intent(inout) :: a
  call sortmdi( n, m, a, 1, n )
end subroutine sortmdi_new



!================================================================================
subroutine argsort1di( n, a, idx )
!================================================================================
  integer, intent(in) :: n
  integer, dimension(n), intent(inout) :: a, idx

  integer :: i, istat
  integer, dimension(:), allocatable :: b
  allocate( b(n), stat=istat )
  if ( istat /= 0 ) then
     print *, 'cannot allocate in argsort1di, n = ', n
     idx = -1
  else
     do i = 1, n
        idx(i) = i
        b(i)   = a(i)
     end do
     call sort1dic( n, b, idx, 1, n )
     deallocate( b )
  end if

end subroutine argsort1di


!================================================================================
subroutine argsortmdi( n, m, a, idx )
!================================================================================

  integer, intent(in) :: n, m
  integer, dimension(m,n), intent(in) :: a
  integer, dimension(n),   intent(inout) :: idx

  integer :: i, istat
  integer, dimension(:,:), allocatable :: b
  allocate( b(m,n), stat=istat )
  if ( istat /= 0 ) then
     print *, 'cannot allocate in argsortmdi, n = ', n
     idx = -1
  else
     do i = 1, n
        idx(i) = i
        b(:,i) = a(:,i)
     end do
     call sortmdic( n, m, b, idx, 1, n )
     deallocate( b )
  end if

end subroutine argsortmdi


!================================================================================
subroutine argsort1dr( n, a, idx )
!================================================================================

  integer n
  real a(n)
  integer idx(n)

  integer :: i, istat
  real, dimension(:), allocatable :: b
  allocate( b(n), stat=istat )
  if ( istat /= 0 ) then
     print *, 'cannot allocate in argsort1dr, n = ', n
     idx = -1
  else
     do i = 1, n
        idx(i) = i
        b(i)   = a(i)
     end do
     call sort1drc( n, b, idx, 1, n )
     deallocate( b )
  end if

end subroutine argsort1dr


!================================================================================
subroutine argsortmdr( n, m, a, idx )
!================================================================================

  integer n, m
  real a(m,n)
  integer idx(n)

  integer :: i, istat
  real :: tol
  real, dimension(:,:), allocatable :: b

  allocate( b(m,n), stat=istat )
  if ( istat /= 0 ) then
     print *, 'cannot allocate in argsort1dr, n = ', n
     idx = -1
  else
     do i = 1, n
        idx(i) = i
        b(:,i) = a(:,i)
     end do
     call sortmdrc( n, m, b, idx, 1, n )
     deallocate( b )
  end if

end subroutine argsortmdr


!================================================================================
subroutine argsort1dd( n, a, idx )
!================================================================================

  integer n
  real(8) a(n)
  integer idx(n)

  integer :: i, istat
  real(8), dimension(:), allocatable :: b
  allocate( b(n), stat=istat )
  if ( istat /= 0 ) then
     print *, 'cannot allocate in argsort1dd, n = ', n
     idx = -1
  else
     do i = 1, n
        idx(i) = i
        b(i)   = a(i)
     end do
     call sort1ddc( n, b, idx, 1, n )
     deallocate( b )
  end if

end subroutine argsort1dd


!================================================================================
subroutine argsortmdd( n, m, a, idx )
!================================================================================

  integer n, m
  real(8) a(m,n)
  integer idx(n)

  integer :: i, istat
  real(8) :: tol
  real(8), dimension(:,:), allocatable :: b

  allocate( b(m,n), stat=istat )
  if ( istat /= 0 ) then
     print *, 'cannot allocate in argsort1dd, n = ', n
     idx = -1
  else
     do i = 1, n
        idx(i) = i
        b(:,i) = a(:,i)
     end do
     call sortmddc( n, m, b, idx, 1, n )
     deallocate( b )
  end if

end subroutine argsortmdd

!================================================================================
subroutine sortmdr_( n, m, a )
!================================================================================
  integer n, m
  real a(m,n)
  call sortmdr( n, m, a, 1, n )
end subroutine sortmdr_

!================================================================================
subroutine sortmdd_( n, m, a )
!================================================================================
  integer n, m
  real(8) a(m,n)
  call sortmdd( n, m, a, 1, n )
end subroutine sortmdd_


!*************************************************************************************



!================================================================================
recursive subroutine sort1di( n, a, l, r )
!================================================================================
  integer,               intent(in)    :: n, l, r
  integer, dimension(n), intent(inout) :: a
  integer left, right, pivot, temp, i, j
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( a(j) < temp ) exit
           a(j+1) = a(j)
           j = j - 1
        end do
        a(j+1) = temp
        i = i + 1
     end do
  else
     pivot = a( l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( pivot < a(right) )
           right = right - 1
        end do
        left = left + 1
        do while ( a(left) < pivot )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(left) ; a(left) = a(right) ; a(right) = temp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sort1di( n, a, l, j-1 )
     call sort1di( n, a, j, r )
  end if

end subroutine sort1di

!================================================================================
recursive subroutine sort1dic( n, a, idx, l, r )
!================================================================================
  integer,               intent(in)    :: n, l, r
  integer, dimension(n), intent(inout) :: a, idx
  integer left, right, pivot, temp, i, j, itemp
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(i) ; itemp = idx(i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( a(j) < temp ) exit
           a(j+1) = a(j) ; idx(j+1) = idx(j)
           j = j - 1
        end do
        a(j+1) = temp ; idx(j+1) = itemp
        i = i + 1
     end do
  else
     pivot = a( l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( pivot < a(right) )
           right = right - 1
        end do
        left = left + 1
        do while ( a(left) < pivot )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(left) ; a(left) = a(right) ; a(right) = temp
           temp = idx(left) ; idx(left) = idx(right) ; idx(right) = temp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sort1dic( n, a, idx, l, j-1 )
     call sort1dic( n, a, idx, j, r )
  end if

end subroutine sort1dic


!================================================================================
recursive subroutine sort1dr( n, a, l, r )
!================================================================================
  integer,               intent(in)    :: n, l, r
  real,    dimension(n), intent(inout) :: a
  integer :: left, right, i, j
  real    :: pivot, temp
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( a(j) < temp ) exit
           a(j+1) = a(j)
           j = j - 1
        end do
        a(j+1) = temp
        i = i + 1
     end do
  else
     pivot = a( l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( pivot < a(right) )
           right = right - 1
        end do
        left = left + 1
        do while ( a(left) < pivot )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(left) ; a(left) = a(right) ; a(right) = temp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sort1dr( n, a, l, j-1 )
     call sort1dr( n, a, j, r )
  end if

end subroutine sort1dr

!================================================================================
recursive subroutine sort1drc( n, a, idx, l, r )
!================================================================================
  integer,               intent(in)    :: n, l, r
  real,    dimension(n), intent(inout) :: a
  integer, dimension(n), intent(inout) :: idx
  integer :: left, right, i, j, itemp
  real    :: pivot, temp
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(i) ; itemp = idx(i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( a(j) < temp ) exit
           a(j+1) = a(j) ; idx(j+1) = idx(j)
           j = j - 1
        end do
        a(j+1) = temp ; idx(j+1) = itemp
        i = i + 1
     end do
  else
     pivot = a( l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( pivot < a(right) )
           right = right - 1
        end do
        left = left + 1
        do while ( a(left) < pivot )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(left) ; a(left) = a(right) ; a(right) = temp
           itemp = idx(left) ; idx(left) = idx(right) ; idx(right) = itemp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sort1drc( n, a, idx, l, j-1 )
     call sort1drc( n, a, idx, j, r )
  end if

end subroutine sort1drc


!================================================================================
recursive subroutine sort1dd( n, a, l, r )
!================================================================================
  integer,               intent(in)    :: n, l, r
  real(8), dimension(n), intent(inout) :: a
  integer :: left, right, i, j
  real(8) :: pivot, temp
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( a(j) < temp ) exit
           a(j+1) = a(j)
           j = j - 1
        end do
        a(j+1) = temp
        i = i + 1
     end do
  else
     pivot = a( l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( pivot < a(right) )
           right = right - 1
        end do
        left = left + 1
        do while ( a(left) < pivot )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(left) ; a(left) = a(right) ; a(right) = temp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sort1dd( n, a, l, j-1 )
     call sort1dd( n, a, j, r )
  end if

end subroutine sort1dd

!================================================================================
recursive subroutine sort1ddc( n, a, idx, l, r )
!================================================================================
  integer,               intent(in)    :: n, l, r
  real(8), dimension(n), intent(inout) :: a
  integer, dimension(n), intent(inout) :: idx
  integer :: left, right, i, j, itemp
  real(8) :: pivot, temp
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(i) ; itemp = idx(i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( a(j) < temp ) exit
           a(j+1) = a(j) ; idx(j+1) = idx(j)
           j = j - 1
        end do
        a(j+1) = temp ; idx(j+1) = itemp
        i = i + 1
     end do
  else
     pivot = a( l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( pivot < a(right) )
           right = right - 1
        end do
        left = left + 1
        do while ( a(left) < pivot )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(left) ; a(left) = a(right) ; a(right) = temp
           itemp = idx(left) ; idx(left) = idx(right) ; idx(right) = itemp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sort1ddc( n, a, idx, l, j-1 )
     call sort1ddc( n, a, idx, j, r )
  end if

end subroutine sort1ddc




!================================================================================
recursive subroutine sortmdi( n, m, a, l, r )
!================================================================================
  integer,                 intent(in)    :: n, m, l, r
  integer, dimension(m,n), intent(inout) :: a
  integer :: left, right, i, j
  integer, dimension(m) :: pivot, temp
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(:,i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( lei( a(:,j), temp, m ) ) exit
           a(:,j+1) = a(:,j)
           j = j - 1
        end do
        a(:,j+1) = temp
        i = i + 1
     end do
  else
     pivot = a(:, l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( lei( pivot, a(:,right), m ) )
           right = right - 1
        end do
        left = left + 1
        do while ( lei( a(:,left), pivot, m ) )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(:,left) ; a(:,left) = a(:,right) ; a(:,right) = temp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sortmdi( n, m, a, l, j-1 )
     call sortmdi( n, m, a, j, r )
  end if

end subroutine sortmdi

!================================================================================
recursive subroutine sortmdic( n, m, a, idx, l, r )
!================================================================================
  integer,                 intent(in)    :: n, m, l, r
  integer, dimension(m,n), intent(inout) :: a
  integer, dimension(n),   intent(inout) :: idx
  integer :: left, right, i, j, itemp
  integer, dimension(m) :: pivot, temp
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(:,i) ; itemp = idx(i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( lei( a(:,j), temp, m ) ) exit
           a(:,j+1) = a(:,j) ; idx(j+1) = idx(j)
           j = j - 1
        end do
        a(:,j+1) = temp ; idx(j+1) = itemp
        i = i + 1
     end do
  else
     pivot = a(:, l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( lei( pivot, a(:,right), m ) )
           right = right - 1
        end do
        left = left + 1
        do while ( lei( a(:,left), pivot, m ) )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(:,left) ; a(:,left) = a(:,right) ; a(:,right) = temp
           itemp = idx(left) ; idx(left) = idx(right) ; idx(right) = itemp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sortmdic( n, m, a, idx, l, j-1 )
     call sortmdic( n, m, a, idx, j, r )
  end if

end subroutine sortmdic


!================================================================================
recursive subroutine sortmdr( n, m, a, l, r )
!================================================================================
  integer,                 intent(in)    :: n, m, l, r
  real,    dimension(m,n), intent(inout) :: a
  integer            :: left, right, i, j
  real, dimension(m) :: pivot, temp
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(:,i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( ler( a(:,j), temp, m ) ) exit
           a(:,j+1) = a(:,j)
           j = j - 1
        end do
        a(:,j+1) = temp
        i = i + 1
     end do
  else
     pivot = a(:, l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( ler( pivot, a(:,right), m ) )
           right = right - 1
        end do
        left = left + 1
        do while ( ler( a(:,left), pivot, m ) )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(:,left) ; a(:,left) = a(:,right) ; a(:,right) = temp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sortmdr( n, m, a, l, j-1 )
     call sortmdr( n, m, a, j, r )
  end if

end subroutine sortmdr

!================================================================================
recursive subroutine sortmdrc( n, m, a, idx, l, r )
!================================================================================
  integer,                 intent(in)    :: n, m, l, r
  real,    dimension(m,n), intent(inout) :: a
  integer, dimension(n),   intent(inout) :: idx
  integer            :: left, right, i, j, itemp
  real, dimension(m) :: pivot, temp
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(:,i) ; itemp = idx(i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( ler( a(:,j), temp, m ) ) exit
           a(:,j+1) = a(:,j) ; idx(j+1) = idx(j)
           j = j - 1
        end do
        a(:,j+1) = temp ; idx(j+1) = itemp
        i = i + 1
     end do
  else
     pivot = a(:, l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( ler( pivot, a(:,right), m ) )
           right = right - 1
        end do
        left = left + 1
        do while ( ler( a(:,left), pivot, m ) )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(:,left) ; a(:,left) = a(:,right) ; a(:,right) = temp
           itemp = idx(left) ; idx(left) = idx(right) ; idx(right) = itemp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sortmdrc( n, m, a, idx, l, j-1 )
     call sortmdrc( n, m, a, idx, j, r )
  end if

end subroutine sortmdrc


!================================================================================
recursive subroutine sortmdd( n, m, a, l, r )
!================================================================================
  integer,                 intent(in)    :: n, m, l, r
  real(8), dimension(m,n), intent(inout) :: a
  integer               :: left, right, i, j
  real(8), dimension(m) :: pivot, temp
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(:,i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( led( a(:,j), temp, m ) ) exit
           a(:,j+1) = a(:,j)
           j = j - 1
        end do
        a(:,j+1) = temp
        i = i + 1
     end do
  else
     pivot = a(:, l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( led( pivot, a(:,right), m ) )
           right = right - 1
        end do
        left = left + 1
        do while ( led( a(:,left), pivot, m ) )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(:,left) ; a(:,left) = a(:,right) ; a(:,right) = temp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sortmdd( n, m, a, l, j-1 )
     call sortmdd( n, m, a, j, r )
  end if

end subroutine sortmdd

!================================================================================
recursive subroutine sortmddc( n, m, a, idx, l, r )
!================================================================================
  integer,                 intent(in)    :: n, m, l, r
  real(8), dimension(m,n), intent(inout) :: a
  integer, dimension(n),   intent(inout) :: idx
  integer               :: left, right, i, j, itemp
  real(8), dimension(m) :: pivot, temp
  if ( r - l < SST ) then
     i = l + 1
     do
        if ( i > r ) exit
        temp = a(:,i) ; itemp = idx(i)
        j = i - 1
        do
           if ( j < l ) exit
           if ( led( a(:,j), temp, m ) ) exit
           a(:,j+1) = a(:,j) ; idx(j+1) = idx(j)
           j = j - 1
        end do
        a(:,j+1) = temp ; idx(j+1) = itemp
        i = i + 1
     end do
  else
     pivot = a(:, l + (r-l)/2 ) ! ~= (l+r)/2, but less overflow-risky
     left  = l - 1 ; right = r + 1
     do while ( left < right )
        right = right - 1
        do while ( led( pivot, a(:,right), m ) )
           right = right - 1
        end do
        left = left + 1
        do while ( led( a(:,left), pivot, m ) )
           left = left + 1
        end do
        if ( left < right ) then
           temp = a(:,left) ; a(:,left) = a(:,right) ; a(:,right) = temp
           itemp = idx(left) ; idx(left) = idx(right) ; idx(right) = itemp
        end if
     end do
     j = left
     if ( left == right ) j = j + 1
     call sortmddc( n, m, a, idx, l, j-1 )
     call sortmddc( n, m, a, idx, j, r )
  end if

end subroutine sortmddc



!======================================================================
function lei( a, b, n )
!======================================================================
! returns .true. if a < b

  integer n
  integer a(n), b(n)
  logical :: lei

  integer :: k

  lei = .true.

  if ( a(1) < b(1) ) return

  if ( a(1) > b(1) ) then
     lei = .false.
     return
  end if

  k = 1
  do
     k = k + 1
     if ( k > n ) exit
     if ( a(k) == b(k) ) cycle
     if ( a(k) > b(k) ) then
        lei = .false.
        return
     end if
     return
  end do
  lei = .false. ! they are fully equal

end function lei


!======================================================================
function ler( a, b, n )
!======================================================================
! returns true if a < b

  integer n
  real a(n), b(n)
  logical :: ler

  integer :: k

  ler = .true.
  if ( a(1) < b(1) ) return
  if ( a(1) > b(1) ) then
     ler = .false.
     return
  end if

  k = 1
  do
     k = k + 1
     if ( k > n ) exit
     if ( a(k) == b(k) ) cycle
     if ( a(k) > b(k) ) then
        ler = .false.
        return
     end if
     return
  end do
  ler = .false. ! they are fully equal

end function ler


!======================================================================
function led( a, b, n )
!======================================================================
! returns true if a < b

  integer n
  real(8) a(n), b(n)
  logical :: led

  integer :: k

  led = .true.
  if ( a(1) < b(1) ) return
  if ( a(1) > b(1) ) then
     led = .false.
     return
  end if

  k = 1
  do
     k = k + 1
     if ( k > n ) exit
     if ( a(k) == b(k) ) cycle
     if ( a(k) > b(k) ) then
        led = .false.
        return
     end if
     return
  end do
  led = .false. ! they are fully equal

end function led


end module sortmod
