module knuth

! This module implements the Fortran 90 version of a PRNG proposed by
! D. E. Knuth.
!
! Available functions
! -------------------
!
! call randomize( seed, rank )
!     where seed is an intent(inout) integer, and rank is an optional integer
!     initializes the PRNG; when seed = 0, a seed is chosen and returned.
!     Use `rank` when parallel instances of this module are used, so that
!     all instances can get a different initialization.
!
!     This routine *must* be called once before using the other functions.
!
! r = rRan()
! i = iRan()
! j = iRanRange( a, b )
!     returns either a double precision pseudo-random number or an integer
!     pseudo-random number (so that a <= j <= b, in the case of iRanRange).


  implicit none

  private

  integer,   parameter :: dpr = selected_real_kind( 14, 300 )
  integer,   parameter :: kk = 100
  integer,   parameter :: ll = 37
  integer,   parameter :: mm = 2**30
  integer,   parameter :: tt = 70
  integer,   parameter :: kkk = kk + kk - 1
  integer,   parameter :: nn = 1024 ! generate RNs nn at a time
  
  character(*), parameter :: ModuleName = 'knuth'

  real(dpr), parameter :: ulp = 1.d0 / ( 2.d0**52 )
  
  real(dpr), dimension(kk) :: ranx
  integer,   dimension(kk) :: iranx

  real(dpr), dimension(nn) :: internal_rns
  integer,   dimension(nn) :: internal_irns
  integer :: ipos, rpos


  public :: rRan, iRan, iRanRange, Randomize, ModuleName, test_knuth

contains

!--------------------------------------------------------------------------------
subroutine Randomize( seed, rank )
!--------------------------------------------------------------------------------
  integer,           intent(inout) :: seed
  integer, optional, intent(in)    :: rank
  integer, dimension(8) :: v

  if ( seed == 0 ) then
     call date_and_time ( values = v )
     seed = 1000 * v(8) + 100 * v(7) + 10 * v(6) + v(8)*v(7)
  end if

  if ( present( rank ) ) seed = 1000 * seed / (3*rank + 1)

  call RNFSTR( seed )
  call rnstrt( seed )
  ipos = nn+1
  rpos = nn+1

end subroutine Randomize



!--------------------------------------------------------------------------------
function rRan( )
!--------------------------------------------------------------------------------

  real(dpr) :: rRan

  if ( rpos > nn ) then
     call rnfarr( internal_rns, nn )
     rpos = 1
  end if  
  rRan = internal_rns( rpos )
  rpos = rpos + 1
end function rRan


!--------------------------------------------------------------------------------
function iRan( )
!--------------------------------------------------------------------------------
  integer :: iRan
  if ( ipos > nn ) then
     call rnarry( internal_irns, nn )
     ipos = 1
  end if
  iRan = internal_irns( ipos )
  ipos = ipos + 1
end function iRan


!--------------------------------------------------------------------------------
function iRanRange( low, high )
!--------------------------------------------------------------------------------
  integer, intent(in) :: low, high
  integer :: iRanRange

  iRanRange = low + floor( (high-low+1._dpr) * rRan() )

end function iRanRange



!--------------------------------------------------------------------------------
subroutine rnfarr( aa, n )
!--------------------------------------------------------------------------------
!       FORTRAN 77 version of "ranf_array"
!       from Seminumerical Algorithms by D E Knuth, 3rd edition (1997)
!       including the MODIFICATIONS made in the 9th printing (2002)
!       ********* see the book for explanations and caveats! *********

  real(dpr), dimension(:), intent(inout) :: aa
  integer,                 intent(in)    :: n

  integer   :: j
  real(dpr) :: y

  do j = 1, kk
     aa(j) = ranx(j)
  end do
  ! do 2
  do j = kk + 1, n
     y = aa(j-kk) + aa(j-ll)
     aa(j) = y - idint(y)
  end do
  ! do 3
  do j = 1, ll
     y = aa(n+j-kk) + aa(n+j-ll)
     ranx(j) = y - idint(y)
  end do
  ! do 4
  do j = ll+1, kk
     y = aa(n+j-kk) + ranx(j-ll)
     ranx(j) = y - idint(y)
  end do

end subroutine rnfarr


!--------------------------------------------------------------------------------
subroutine rnfstr( seed )
!--------------------------------------------------------------------------------

  integer, intent(in) :: seed

  integer   :: s, sseed, j, t
  real(dpr) :: ss, v
  real(dpr), dimension(kkk) :: u
  
  if ( seed < 0 ) then
     sseed = mm - 1 - mod(-1-seed,mm)
  else
     sseed = mod(seed,mm)
  end if
  ss = 2._dpr * ulp * (sseed+2)
  do j = 1, kk
     u(j) = ss
     ss = ss + ss
     if ( ss >= 1._dpr ) ss = ss - 1._dpr + 2._dpr * ulp
  end do
  u(2) = u(2) + ulp
  s = sseed
  t = tt - 1
  10 continue
  do j = kk, 2, -1
     u(j+j-1) = u(j)
     u(j+j-2) = 0
  end do
  do j = kkk, kk+1, -1
     v = u(j-(kk-ll)) + u(j)
     u(j-(kk-ll)) = v - idint(v)
     v = u(j-kk) + u(j)
     u(j-kk) = v - idint(v)
  end do
  if ( mod(s,2) == 1 ) then
     do j = kk, 1, -1
        u(j+1) = u(j)
     end do
     u(1) = u(kk+1)
     v = u(ll+1) + u(kk+1)
     u(ll+1) = v - idint(v)
  end if

  if ( s /= 0 ) then
     s = s / 2
  else
     t = t - 1
  end if

  if ( t > 0 ) goto 10

  do j = 1, ll
     ranx(j+kk-ll) = u(j)
  end do

  do j = ll+1, kk
     ranx(j-ll) = u(j)
  end do
  
  do j = 1, 10
     call rnfarr(u,kkk)
  end do

end subroutine rnfstr


!--------------------------------------------------------------------------------
subroutine rnarry( aa, n )
!--------------------------------------------------------------------------------
!       FORTRAN 77 version of "ran_array"
!       from Seminumerical Algorithms by D E Knuth, 3rd edition (1997)
!       including the MODIFICATIONS made in the 9th printing (2002)
!       ********* see the book for explanations and caveats! *********

  integer, dimension(:), intent(inout) :: aa
  integer,               intent(in)    :: n

  integer :: j

  do j = 1, kk
     aa(j) = iranx(j)
  end do
  do j = kk+1, n
     aa(j) = aa(j-kk) - aa(j-ll)
     if ( aa(j) < 0 ) aa(j) = aa(j) + mm
  end do
  do j = 1, ll
     iranx(j) = aa(n+j-kk) - aa(n+j-ll)
     if ( iranx(j) < 0 ) iranx(j) = iranx(j) + mm
  end do
  do j = ll+1, kk
     iranx(j) = aa(n+j-kk) - iranx(j-ll)
     if ( iranx(j) < 0 ) iranx(j) = iranx(j) + mm
  end do

end subroutine rnarry


!--------------------------------------------------------------------------------
subroutine rnstrt( seed )
!--------------------------------------------------------------------------------

  integer, intent(inout) :: seed

  integer, dimension(kkk) :: x
  integer                 :: j, sseed, ss, t

  IF (SEED .LT. 0) THEN
     SSEED=MM-1-MOD(-1-SEED,MM)
  ELSE
     SSEED=MOD(SEED,MM)
  END IF
  SS=SSEED-MOD(SSEED,2)+2
  DO J=1,KK
     X(J)=SS
     SS=SS+SS
     IF (SS .GE. MM) SS=SS-MM+2
  end do
  X(2)=X(2)+1
  SS=SSEED
  T=TT-1
10 continue
  DO J=KK,2,-1
     X(J+J-1)=X(J)
     X(J+J-2)=0
  end do
  DO J=KKK,KK+1,-1
     X(J-(KK-LL))=X(J-(KK-LL))-X(J)
     IF (X(J-(KK-LL)) .LT. 0) X(J-(KK-LL))=X(J-(KK-LL))+MM
     X(J-KK)=X(J-KK)-X(J)
     IF (X(J-KK) .LT. 0) X(J-KK)=X(J-KK)+MM
  end do
  IF (MOD(SS,2) .EQ. 1) THEN
     DO J=KK,1,-1
        X(J+1)=X(J)
     end do
     X(1)=X(KK+1)
     X(LL+1)=X(LL+1)-X(KK+1)
     IF (X(LL+1) .LT. 0) X(LL+1)=X(LL+1)+MM
  END IF
  IF (SS .NE. 0) THEN
     SS=SS/2
  ELSE
     T=T-1
  END IF
  IF (T .GT. 0) GO TO 10
  DO J=1,LL
     iranx(J+KK-LL)=X(J)
  end do
  DO J=LL+1,KK
     iranx(J-LL)=X(J)
  end do
  DO J=1,10
     CALL RNARRY(X,KKK)
  end do
  
end subroutine rnstrt



!================================================================================
subroutine test_knuth()
!================================================================================

  real(8), dimension(2009) :: a
  integer :: seed

  integer :: i, j
  real(8) :: r
  real    :: t1, t2

  print *

  CALL RNFSTR(310952)
  DO I=1,2010
     CALL RNFARR(A,1009)
  end do
  r = 0.36410514377569680455_8
  if ( a(1) == r ) then
     print *, 'value ok'
  else
     print *, 'value NOT OK', a(1), 0.36410514377569680455_dpr
  end if

  seed = 310952
  call randomize( seed )

  call cpu_time( t1 )
  do i = 1, 100000000
     r = rRan()
  end do
  call cpu_time( t2 )
  print *, 'r = ', r
  print *, 'elapsed = ', t2 - t1

end subroutine test_knuth

end module knuth


!!$PROGRAM MAIN
!!$!      a rudimentary test program:
!!$  use knuth
!!$  IMPLICIT none
!!$  real(dpr), dimension(2009) :: A
!!$  integer :: i
!!$  CALL RNFSTR(310952)
!!$  DO I=1,2010
!!$     CALL RNFARR(A,1009)
!!$  end do
!!$  PRINT '(F22.20,a,F22.20)',A(1), ' should be ', 0.36410514377569680455_dpr
!!$!                   the number should be 0.36410514377569680455
!!$  CALL RNFSTR(310952)
!!$  DO I=1,1010
!!$          CALL RNFARR(A,2009)
!!$  end do
!!$  PRINT '(F22.20,a)',A(1), ' (should be the same)'
!!$!                                 again, 0.36410514377569680455
!!$END PROGRAM MAIN
